<#
        .SYNOPSIS
        Runs selected Eric Zimmerman tools against a collection of mounted disk images.

        .DESCRIPTION
        Runs selected Eric Zimmerman tools against a collection of mounted disk images. These include AmcacheParser, AppCompatCacheParser, MFTECmd, PECmd, and also EvtxECmd by default. The Zimmerman Tools must be downloaded and extracted to a folder on the local machine. 
        
        .PARAMETER MountDirectory
        The folder where your triage disk images are mounted. Each image should be mounted to a subfolder name. This is handled automatically by Mount-TriageVhdx.ps1.
        
        .PARAMETER ZimmermanToolsDirectory
        Specifies the directory containing the Zimmerman Tools. This folder will often be named 'net6' and contains multiple executables and folders. This script will recursively search the provided folder for executables matching the expected names.

        .PARAMETER OutDirectory
        The folder where your output CSV files will be stored. The folder must already exist. 

        .PARAMETER IncludeEvtxECmd
        An optional parameter that specifies whether to include EvtxECmd.exe in the processing. The default is $true. 

        .PARAMETER StartDateTime
        An optional parameter that specifies the start date and time to use for filtering the logs. The date and time must be in the format yyyy-MM-dd HH:mm:ss zzz. The time zone must be specified in the format +/-HH:mm. The time zone is optional. If not specified, the local time zone will be used. If specified, only events after the specified date and time will be included in the output. 

        .PARAMETER EndDateTime
        An optional parameter that specifies the end date and time to use for filtering the logs. The date and time must be in the format yyyy-MM-dd HH:mm:ss zzz. The time zone must be specified in the format +/-HH:mm. The time zone is optional. If not specified, the local time zone will be used. If specified, only events before the specified date and time will be included in the output. 

        .PARAMETER ReduceEvtxECmdOutput
        An optional switch that specifies whether to reduce the amount of output from EvtxECmd. If specified, only the following event IDs will be processed 4720, 4728, 4732, 4756, 4768, 4769, 4770, 4771, 4776, 4624, 4625, 4648, 4672, 4778, 4779, 1024, 1102, 98, 131, 5140, 5145, 106, 140, 141, 200, 201, 4698, 1102, 104, 7045, 4697, 5861, 1006, 1007, 1013, 1116, 1117, 1118, 1119, 5004, 5007, 5001, 5012, 4103, 4104, 400, 800. The default is $false.

        .PARAMETER Confirm
        An optional switch that specifies whether to prompt the user to confirm that they want to continue. If not specified, the user will not be prompted. If specified, the user will be provided an overview of the actions that will be taken and need to confirm before processing begins.

        .PARAMETER CombineEvtx
        An optional switch that specifies whether to combine the output from EvtxECmd into a single CSV file. The default is $false. This is only supported if -StartDateTime, -EndDateTime, or -ReduceEvtxECmdOutput is used. If specified, the output will be combined into a single file named CombinedEvtx-Filtered.csv stored in the OutDirectory. Use with caution to avoid creating a file that is too big to open or process.

        .EXAMPLE
        .\Run-ZimmermanTools.ps1 -MountDirectory "E:\MountDir\" -ZimmermanToolsDirectory "C:\Tools\ZimmermanTools\net6\" -OutDirectory "F:\OutDir"
        Runs the script with the minimal parameters. This will process event logs with EvtxECmd as well as the other tools. 
        
        .EXAMPLE
        .\Run-ZimmermanTools.ps1 -MountDirectory "E:\MountDir\"" -ZimmermanToolsDirectory "C:\Tools\ZimmermanTools\net6\" -OutDirectory "F:\OutDir" -IncludeEvtxECmd $false
        This example shows how to run the script without processing event logs with EvtxECmd.

        .NOTES
        Written by Steve Anson and Dr. Sorot Panichprecha. This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. The GNU General Public License can be found at <https://www.gnu.org/licenses/>.
        
        .LINK
        You can download Eric Zimmerman's tools from here: https://ericzimmerman.github.io/#!index.md

        .LINK
        You can download a script that will download all of Eric Zimmerman's tools from here: https://github.com/EricZimmerman/Get-ZimmermanTools

        .LINK
        You can learn more about high-value Event IDs in this reference https://www.appliedincidentresponse.com/files/Windows-Event-Log-Analyst-Reference.pdf
    #>

    param (
        [Parameter(Position=0,mandatory=$true)]
        $MountDirectory,

        [Parameter(Position=1,mandatory=$true)]
        $ZimmermanToolsDirectory,

        [Parameter(Position=2,mandatory=$true)]
        $OutDirectory,

        [Parameter(Position=3,mandatory=$false)]
        [ValidateSet($true, $false)]
        $IncludeEvtxECmd = $true,  

        [Parameter(mandatory=$false)]
        $StartDateTime = $null,

        [Parameter(mandatory=$false)]
        $EndDateTime = $null,

        [Parameter(mandatory=$false)]
        [switch]$ReduceEvtxECmdOutput,

        [Parameter(mandatory=$false)]
        [switch]$Confirm,

        [Parameter(mandatory=$false)]
        [switch]$CombineEvtx
    )
    
    # Check if the required directories and files exist
    if (-not (Test-Path $MountDirectory)) {
        Write-Host "Mount directory '$MountDirectory' does not exist. Exiting..."
        exit 1
    }
    
    if (-not (Test-Path $ZimmermanToolsDirectory)) {
        Write-Host "Zimmerman Tools directory '$ZimmermanToolsDirectory' does not exist. Exiting..."
        exit 1
    }

    if(-not ($IncludeEvtxECmd)) {
        if ($StartDateTime -or $EndDateTime) {
            Write-Host "-StartDateTime and -EndDateTime are not supported with -IncludeEvtxECmd=`$False. Exiting..."
            exit 1
        }
    }

    # If present, verify that the StartDateTime and EndDateTime parameters are in the correct format
    if ($StartDateTime) {
        try {
            $parsedStartDateTime = [DateTimeOffset]::Parse($StartDateTime)
            $parsedStartDateTime = [DateTimeOffset]::Parse($StartDateTime).ToUniversalTime()
        }
        catch {
            Write-Host "Invalid format for StartDateTime. Please enter a valid date and time string."
            Write-Host "Accepted format: yyyy-MM-dd HH:mm:ss zzz"
            Write-Host "Example: -StartDateTime '2022-01-01 09:00:00 +03:00'"
            exit
        }
    }

    if ($EndDateTime) {
        try {
            $parsedEndDateTime = [DateTimeOffset]::Parse($EndDateTime)
            $parsedEndDateTime = [DateTimeOffset]::Parse($EndDateTime).ToUniversalTime()
        }
        catch {
            Write-Host "Invalid format for EndDateTime. Please enter a valid date and time string."
            Write-Host "Accepted format: yyyy-MM-dd HH:mm:ss zzz"
            Write-Host "Example: -EndDateTime '2022-01-01 09:00:00 +03:00'"

            exit
        }
    }

    # Check if the CombineEvtx switch is used without the required parameters
    If ($CombineEvtx) {
        if (-not $StartDateTime -and -not $EndDateTime -and -not $ReduceEvtxECmdOutput) {
            Write-Host "-CombineEvtx is only supported if -StartDateTime, -EndDateTime, or -ReduceEvtxECmdOutput is used. Exiting..."
            exit 1
        }
    }

    # Display the mount points that will be processed
    $volumes = Get-WmiObject -Class Win32_Volume | Where-Object { $_.DriveType -eq 3 -and $_.Name -like "*$MountDirectory*" }
    Write-Host "The following mounted image locations will be processed:"
    foreach ($volume in $volumes) {
        Write-Host "$($volume.Name)"
    }

    # Display the options that will be used for processing with EvtxECmd
    If ($IncludeEvtxECmd) {
        if ($ReduceEvtxECmdOutput) {
            Write-Host "EvtxECmd will be used to process only the following event IDs: 4720, 4728, 4732, 4756, 4768, 4769, 4770, 4771, 4776, 4624, 4625, 4648, 4672, 4778, 4779, 1024, 1102, 98, 131, 5140, 5145, 106, 140, 141, 200, 201, 4698, 1102, 104, 7045, 4697, 5861, 1006, 1007, 1013, 1116, 1117, 1118, 1119, 5004, 5007, 5001, 5012, 4103, 4104, 400, 800"
        }
        else {
            Write-Host "EvtxECmd will be used to process all event IDs"
        }
    }
    else {
        Write-Host "EvtxECmd will not be used to process event logs"
    }

    # Display the times that will be used for processing
    if ($StartDateTime) {
        if ($StartDateTime) {
            Write-Host "The following start date and time will be used for processing with EvtxECmd:"
            Write-Host "Start Year: $($parsedStartDateTime.Year)"
            Write-Host "Start Month: $($parsedStartDateTime.ToString("MMM"))"
            Write-Host "Start Day: $($parsedStartDateTime.Day)"
            Write-Host "Start Time: $($parsedStartDateTime.ToString("HH:mm:ss"))"
            Write-Host "Start Time Zone: $($parsedStartDateTime.Offset)"
        }
    }
    else {
        Write-Host "Processsing with EvtxECmd will start at the beginning of the logs with no start date filter"
    }
    if ($EndDateTime) {
        Write-Host "The following end date and time will be used for processing with EvtxECmd:"
        Write-Host "End Year: $($parsedEndDateTime.Year)"
        Write-Host "End Month: $($parsedEndDateTime.ToString("MMM"))"
        Write-Host "End Day: $($parsedEndDateTime.Day)"
        Write-Host "End Time: $($parsedEndDateTime.ToString("HH:mm:ss"))"
        Write-Host "End Time Zone: $($parsedEndDateTime.Offset)"
    }
    else {
        Write-Host "Processsing with EvtxECmd will stop at the end of the logs with no end date filter"
    }

    # Confirm that the user wants to continue if requested
    if ($Confirm) {
        while ($true) {
            $response = Read-Host "Do you want to continue? (Y/N)"
            if ($response -imatch "^(y|yes)$") {
                break
            }
            elseif ($response -imatch "^(N|No)$") {
                Write-Host "Exiting..."
                exit
            }
            else {
                Write-Host "Invalid response. Please enter Y or N."
            }
        }
    }

    # Declare script variables
    $script:processes = @()

    # Recursively search through $ZimmermanToolsDirectory to find each executable, exit if required executables are missing
    $EvtxeCmdExe = Get-ChildItem -Path $ZimmermanToolsDirectory -Recurse -Filter "EvtxeCmd.exe" | Select-Object -ExpandProperty FullName
    $AmcacheParserExe = Get-ChildItem -Path $ZimmermanToolsDirectory -Recurse -Filter "AmcacheParser.exe" | Select-Object -ExpandProperty FullName
    $AppCompatCacheParserExe = Get-ChildItem -Path $ZimmermanToolsDirectory -Recurse -Filter "AppCompatCacheParser.exe" | Select-Object -ExpandProperty FullName
    $MFTECmdExe = Get-ChildItem -Path $ZimmermanToolsDirectory -Recurse -Filter "MFTECmd.exe" | Select-Object -ExpandProperty FullName
    $PECmdExe = Get-ChildItem -Path $ZimmermanToolsDirectory -Recurse -Filter "PECmd.exe" | Select-Object -ExpandProperty FullName

    $missingExecutables = @()
    if (-not $EvtxeCmdExe) { $missingExecutables += "EvtxeCmd.exe" }
    if (-not $AmcacheParserExe) { $missingExecutables += "AmcacheParser.exe" }
    if (-not $AppCompatCacheParserExe) { $missingExecutables += "AppCompatCacheParser.exe" }
    if (-not $MFTECmdExe) { $missingExecutables += "MFTECmd.exe" }
    if (-not $PECmdExe) { $missingExecutables += "PECmd.exe" }

    if ($missingExecutables) {
        Write-Host "The following executables are missing:"
        $missingExecutables | ForEach-Object {
            Write-Host $_
        }
        exit 1
    }

    #Declare Function to run Zimmerman Tools
    function RunZimmermanTools {
        param (
            $HostDir
        )

        # Run EvtxeCmd
        If ($IncludeEvtxECmd) {
            If ($StartDateTime -or $EndDateTime -or $ReduceEvtxECmdOutput) {
                $EvtxECmdOutputFile = "evtxecmd-filtered-" + $HostDir + ".csv"
            }
            else {
                $EvtxECmdOutputFile = "evtxecmd-" + $HostDir + ".csv"
            }
            $EvtxDir = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\Windows\System32\winevt\logs"
            $EvtxECmdAguments = "-d `"$EvtxDir`" --csv `"$($OutDirectory)$($HostDir)`" --csvf `"$EvtxECmdOutputFile`""
            if ($StartDateTime) {
                $EvtxECmdAguments += " --sd `"$($parsedStartDateTime.ToString('yyyy-MM-dd HH:mm:ss'))`""
            }
            if ($EndDateTime) {
                $EvtxECmdAguments += " --ed `"$($parsedEndDateTime.ToString('yyyy-MM-dd HH:mm:ss'))`""
            }
            if ($ReduceEvtxECmdOutput) {
                $EvtxECmdAguments += " --inc 4720,4728,4732,4756,4768,4769,4770,4771,4776,4624,4625,4648,4672,4778,4779,1024,1102,98,131,5140,5145,106,140,141,200,201,4698,1102,104,7045,4697,5861,1006,1007,1013,1116,1117,1118,1119,5004,5007,5001,5012,4103,4104,400,800"
            }
            $process = Start-Process -PassThru -FilePath $EvtxeCmdExe -ArgumentList $EvtxECmdAguments
            $script:processes += $process
        }

        # Run AmcacheParser 
        $AmcacheOutputFile = "amcache-" + $HostDir + ".csv"
        $AmcachePath = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\Windows\AppCompat\Programs\Amcache.hve"
        $process = Start-Process -PassThru -FilePath $AmcacheParserExe -ArgumentList "-f `"$AmcachePath`" --csv `"$($OutDirectory)$($HostDir)`" --csvf `"$AmcacheOutputFile`""
        $script:processes += $process

        # Run AppCompatCacheParser
        $AppCompatCachePath = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\Windows\System32\config\SYSTEM"
        $AppCompatCacheOutputFile = "appcompatcache-" + $HostDir + ".csv"
        $process = Start-Process -PassThru -FilePath $AppCompatCacheParserExe -ArgumentList "-f `"$AppCompatCachePath`" --csv `"$($OutDirectory)$($HostDir)`" --csvf `"$AppCompatCacheOutputFile`""
        $script:processes += $process

        # Run MFTECmd for the MFT
        $MFTECmdOutputFile = "mftecmd-mft-" + $HostDir + ".csv"
        $MFTPath = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\`$MFT"
        $process = Start-Process -PassThru -FilePath $MFTECmdExe -ArgumentList "-f `"$MFTPath`" --csv `"$($OutDirectory)$($HostDir)`" --csvf `"$MFTECmdOutputFile`""
        $script:processes += $process

        # Run MFTECmd for the USN Journal
        $JournalOutputFile = "mftecmd-usnjournal-" + $HostDir + ".csv"
        $JournalPath = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\`$Extend\`$J"
        $process = Start-Process -PassThru -FilePath $MFTECmdExe -ArgumentList "-m `"$MFTPath`" -f `"$JournalPath`" --csv `"$($OutDirectory)$($HostDir)`" --csvf `"$JournalOutputFile`""
        $script:processes += $process
           
        #Run PECmd
        $PECmdOutputFile = "pecmd-" + $HostDir + ".csv"
        $PrefetchPath = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\Windows\prefetch"
        $process = Start-Process -PassThru -FilePath $PECmdExe -ArgumentList "-d `"$PrefetchPath`" --csv `"$($OutDirectory)$($HostDir)`" --csvf `"$PECmdOutputFile`""
        $script:processes += $process         
    }

    # Main program
    #Iterate through all mounted image directories and process data 
    Write-Host "Now processing all images mounted in $MountDirectory"
    $HostDir = Get-ChildItem -Path "$MountDirectory" -Directory -Name
    $HostDir | ForEach-Object {
    RunZimmermanTools -HostDir $_
    }

    Write-Host "Each new window is processing a task. `nAs each task completes, the associated window will close."
    Write-Host "Until then, the data is still being processed. Please wait..."

    # Monitor processes and update progress bar
    while ($script:processes | Where-Object { $_.HasExited -eq $false }) {
        Start-Sleep -Seconds 1 # Wait before checking again

        # Calculate completed processes
        $completed = ($script:processes | Where-Object { $_.HasExited }).Count + ($script:SrumProcesses | Where-Object { $_.HasExited }).Count
        $total = $script:processes.Count + $script:SrumProcesses.Count
        $percentComplete = ($completed / $total) * 100

        Write-Progress -Activity "Processing Data" -Status "Waiting for processing to complete" -PercentComplete $percentComplete
    }

    # After processing, combine the output from EvtxECmd into a single file if requested
    If ($CombineEvtx) {
        if (Test-Path "$OutDirectory\CombinedEvtx-Filtered.csv") {
            Write-Host -ForegroundColor Red "ERROR: A file named CombinedEvtx-Filtered.csv already exists in the output directory.`nSkipping creation of new combined file."
            }
            else{
                Write-Host "Creating combined EvtxECmd at $OutDirectory\CombinedEvtx-Filtered.csv"
                $files = Get-ChildItem -Path $OutDirectory -Filter "evtxecmd-filtered*" -Recurse

                foreach ($file in $files) {
                    $content = Get-Content $file.FullName
                    if ($file -ne $files[0]) {
                        $content = $content | Select-Object -Skip 1
                    }
                    $content | Out-File -FilePath "$OutDirectory\CombinedEvtx-Filtered.csv" -Append
                }
            }
    }

    # Report completion
    Write-Host -ForegroundColor Green "Processing is completed.`nThe results are stored in $($OutDirectory) with a subfolder for each system."
    Write-Host "You may now begin your analysis."
    