<#
        .SYNOPSIS
        Ingests Windows event logs from disk images mounted with Mount-TriageVhdx.ps1 into Elasticsearch.

        .DESCRIPTION
        Ingests Windows event logs from disk images mounted with Mount-TriageVhdx.ps1 into Elasticsearch. Must provide the details of your Elastic implementation in the YAML file. Elastic must also have the Winlogbeat components installed. This can be done, after the YAML file is configured, by running .\winlogbeat.exe -E COMPUTER_NAME="Setup" setup -e 
        
        .PARAMETER MountDirectory
        The folder where your triage disk images are mounted. Each image should be mounted to a subfolder name. This is handled automatically by Mount-TriageVhdx.ps1.
        
        .PARAMETER WinlogbeatDirectory
        Specifies the directory where winlogbeat.exe and the associated YAML file are stored. Both must be in the root of the specified directory. The YAML file must be modified to contain the details of your Elastic implementation.
        
         .PARAMETER YamlFile
        Specifies the name of the winlogbeat YAML file to use. Defaults to winlogbeat.yml if not specified.
        
        .EXAMPLE
        PS> .\Run-Winlogbeat.ps1 -MountDirectory "D:\MountedImagesFolder\" -WinlogbeatDirectory "D:\Winlogbeat"
        The above syntax will run the script with the only required parameters.

        .EXAMPLE
        PS> .\Run-Winlogbeat.ps1 -MountDirectory "D:\MountedImagesFolder\" -WinlogbeatDirectory "D:\Winlogbeat" -YamlFile "winlogbeat-forensics.yml"
        The above syntax will run the script with the required parameters and a custom YAML file name.

        .NOTES
        Written by Dr. Sorot Panichprecha and Steve Anson. This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. The GNU General Public License can be found at <https://www.gnu.org/licenses/>.
       
        .LINK
        You can download Winlogbeats from here: https://www.elastic.co/downloads/beats/winlogbeat

        .LINK
        You find instructions to build an Elastic stack in Docker here: https://www.elastic.co/guide/en/elasticsearch/reference/current/docker.html
     
    #>


    param (
    [Parameter(Position=0,mandatory=$true)]
    $MountDirectory,
    [Parameter(Position=1,mandatory=$true)]
    $WinlogbeatDirectory,
    [Parameter(Position=2,mandatory=$false)]
    $YamlFile = "winlogbeat.yml"
)

# Check if the required directories and files exist
if (-not (Test-Path $MountDirectory)) {
    Write-Host "Mount directory '$MountDirectory' does not exist. Exiting..."
    exit
}

if (-not (Test-Path $WinlogbeatDirectory)) {
    Write-Host "Winlogbeat directory '$WinlogbeatDirectory' does not exist. Exiting..."
    exit
}

if (-not (Test-Path "$WinlogbeatDirectory\$YamlFile")) {
    Write-Host "Winlogbeat YAML file '$WinlogbeatDirectory$YamlFile' does not exist. Exiting..."
    exit
}

if (-not (Test-Path "$WinlogbeatDirectory\winlogbeat.exe")) {
    Write-Host "'$WinlogbeatDirectory\winlogbeat.exe' does not exist.`nYou can download it from https://www.elastic.co/downloads/beats/winlogbeat `nExiting..."
    exit
}

# Parse each mounted image in separate process
Write-Host "Now processing all images mounted in $MountDirectory using the configuration specified in $WinlogbeatDirectory$YamlFile"
$HostDir = Get-ChildItem -Path "$MountDirectory" -Directory -Name
$script:processes = @()
$HostDir | ForEach-Object {
    function RunWinlogBeat {
        param (
            $HostDir
        )
        
        $EvtxDir = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\Windows\System32\winevt\logs"
        Write-Host "Opening a new window to process logs in $($EvtxDir)"
        $process = Start-Process -PassThru -FilePath "$WinlogbeatDirectory\winlogbeat.exe" -ArgumentList "-c $WinlogbeatDirectory\$YamlFile -E EVTX_DIR=$($EvtxDir) -E COMPUTER_NAME=$($HostDir)"
        $script:processes += $process 
    }
    
    RunWinlogBeat -HostDir $_
}

Write-Host "****************************************************************************************************"
Write-Host "Each new window is processing a system. `nAs each system's processing completes, its window will close. `nUntil then, the data is still being added to Elastic. Please wait..."


# Monitor processes and update progress bar
while ($script:processes | Where-Object { $_.HasExited -eq $false }) {
    Start-Sleep -Seconds 1 # Wait before checking again

    # Calculate completed processes
    $completed = ($script:processes | Where-Object { $_.HasExited }).Count
    $total = $script:processes.Count
    $percentComplete = ($completed / $total) * 100

    Write-Progress -Activity "Processing Logs" -Status "Waiting for processing to complete" -PercentComplete $percentComplete
}

# After all processes have completed, report completion
Write-Host "All instances of winlogbeat.exe have completed processing."
Write-Host "The $($WinlogbeatDirectory)logs folder contains Winlogbeat log data for each system. `nCheck there for any errors. You may now begin your analysis."