<#
        .SYNOPSIS
        Uses Hayabusa to analyze Windows event logs found in disk images mounted with Mount-TriageVhdx.ps1.

        .DESCRIPTION
        Uses Hayabusa to analyze Windows event logs found in disk images mounted with Mount-TriageVhdx.ps1 with Hayabusa. Outputs one CSV per host with all detections found and another for high and critical detections if desired. Provides a combined CSV from all hosts with high and critical detections. 
        
        .PARAMETER MountDirectory
        The folder where your triage disk images are mounted. Each image should be mounted to a subfolder name. This is handled automatically by Mount-TriageVhdx.ps1.
        
        .PARAMETER HayabusaDirectory
        Specifies the directory containing the Hayabusa exectuable. The .exe must be in the root of the specified directory.

        .PARAMETER OutDirectory
        The folder where your output CSV files will be stored. The folder must already exist. 

        .PARAMETER HighCrit
        An optional parameter that specifies whether to also output separate CSVs containing only high and critical detections for each system. A combined-high+crit.csv will also be created for all systems. Using this option provides a quick way to see the high and critical findings, but will increase processing time. Defaults to true if not specified. Either way, the high and critical detection results are also included in the default processing of all rules.

        .PARAMETER StartDateTime
        An optional parameter that specifies the start date and time to use for filtering the logs. The date and time must be in the format yyyy-MM-dd HH:mm:ss zzz. The time zone must be specified in the format +/-HH:mm. The time zone is optional. If not specified, the local time zone will be used. If specified, only events after the specified date and time will be included in the output. 

        .PARAMETER EndDateTime
        An optional parameter that specifies the end date and time to use for filtering the logs. The date and time must be in the format yyyy-MM-dd HH:mm:ss zzz. The time zone must be specified in the format +/-HH:mm. The time zone is optional. If not specified, the local time zone will be used. If specified, only events before the specified date and time will be included in the output. 

        .PARAMETER Confirm
        An optional switch that specifies whether to prompt the user to confirm that they want to continue. If not specified, the user will not be prompted. If specified, the user will be provided an overview of the actions that will be taken and need to confirm before processing begins.

        .EXAMPLE
        PS> Run-Hayabusa.ps1 -MountDirectory "D:\MountedImagesFolder\" -HayabusaDirectory "D:\Hayabusa" -OutDirectory "D:\HayabusaOutput\"
        This will run the script with only required parameters, and default to also generating a summary of high and critical detections.
        
         .EXAMPLE
        PS> Run-Hayabusa.ps1 -MountDirectory "D:\MountedImagesFolder\" -HayabusaDirectory "D:\Hayabusa" -OutDirectory "D:\HayabusaOutput\" -HighCrit $false
        This will run the script with the required parameters, but not generate a summary of high and critical detections. 
        
        .EXAMPLE
        PS> Run-Hayabusa.ps1 -MountDirectory "D:\MountedImagesFolder\" -HayabusaDirectory "D:\Hayabusa" -OutDirectory "D:\HayabusaOutput\" -StartDateTime "2022-01-01 09:00:00 +03:00" -EndDateTime "2022-01-01 09:00:00 +03:00"
        This will run the script with the required parameters, but filter the results to only include events between the specified start and end date and time. The date and time must be in the format yyyy-MM-dd HH:mm:ss zzz. The time zone must be specified in the format +/-HH:mm. The time zone is optional. If not specified, the local time zone will be used.

        .EXAMPLE
        PS> Run-Hayabusa.ps1 -MountDirectory "D:\MountedImagesFolder\" -HayabusaDirectory "D:\Hayabusa" -OutDirectory "D:\HayabusaOutput\" -EndDateTime "2022-01-01 09:00:00 +03:00"
        This will run the script with the required parameters, but filter the results to only include events before the specified end date and time. The date and time must be in the format yyyy-MM-dd HH:mm:ss zzz. The time zone must be specified in the format +/-HH:mm. The time zone is optional. If not specified, the local time zone will be used. 
        
        .EXAMPLE
        PS> Run-Hayabusa.ps1 -MountDirectory "D:\MountedImagesFolder\" -HayabusaDirectory "D:\Hayabusa" -OutDirectory "D:\HayabusaOutput\" -StartDateTime "2022-01-01 09:00:00 +03:00" 
        This will run the script with the required parameters, but filter the results to only include events after the specified start date and time. The date and time must be in the format yyyy-MM-dd HH:mm:ss zzz. The time zone must be specified in the format +/-HH:mm. The time zone is optional. If not specified, the local time zone will be used. 

        .NOTES
        Written by Steve Anson and Dr. Sorot Panichprecha. This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. The GNU General Public License can be found at <https://www.gnu.org/licenses/>.
        
        .LINK
        You can download Hayabusa from here: https://github.com/Yamato-Security/hayabusa/releases
    #>


    param (
        [Parameter(Position=0,mandatory=$true)]
        $MountDirectory,
        [Parameter(Position=1,mandatory=$true)]
        $HayabusaDirectory,
        [Parameter(Position=2,mandatory=$true)]
        $OutDirectory,
        [Parameter(mandatory=$false)]
        $HighCrit = $true,
        [Parameter(mandatory=$false)]
        $StartDateTime = $null,
        [Parameter(mandatory=$false)]
        $EndDateTime = $null,
        [Parameter(mandatory=$false)]
        [switch]$Confirm
    )
    
    # Verify that the required directories and files exist
    if (-not (Test-Path $MountDirectory)) {
        Write-Host "Mount directory '$MountDirectory' does not exist. Exiting..."
        exit
    }

    if (-not (Get-ChildItem -Path $MountDirectory)) {
        Write-Host "Mount directory '$MountDirectory' is empty. Exiting..."
        exit
    }

    $volumes = Get-WmiObject -Class Win32_Volume | Where-Object { $_.DriveType -eq 3 -and $_.Name -like "*$MountDirectory*" }
    if ($volumes.Count -eq 0) {
        Write-Host "No mount points found for '$MountDirectory'. Exiting..."
        exit
    }

    if (-not (Test-Path $HayabusaDirectory)) {
        Write-Host "Hayabusa directory '$HayabusaDirectory' does not exist. Exiting..."
        exit
    }
    
    if (-not (Test-Path "$HayabusaDirectory\hayabusa*.exe")) {
        Write-Host "'$HayabusaDirectory\' does not contain a hayabusa executable.`nYou can download it from https://github.com/Yamato-Security/hayabusa/releases `nExiting..."
        exit
    }
    # Verify date and time format if specified
    if ($StartDateTime) {
        try {
            $parsedStartDateTime = [DateTimeOffset]::Parse($StartDateTime)
        }
        catch {
            Write-Host "Invalid format for StartDateTime. Please enter a valid date and time string."
            Write-Host "Accepted format: yyyy-MM-dd HH:mm:ss zzz"
            Write-Host "Example: -StartDateTime '2022-01-01 09:00:00 +03:00'"
            exit
        }
    }

    if ($EndDateTime) {
        try {
            $parsedEndDateTime = [DateTimeOffset]::Parse($EndDateTime)
        }
        catch {
            Write-Host "Invalid format for EndDateTime. Please enter a valid date and time string."
            Write-Host "Accepted format: yyyy-MM-dd HH:mm:ss zzz"
            Write-Host "Example: -EndDateTime '2022-01-01 09:00:00 +03:00'"

            exit
        }
    }
   
    # Display the mount points that will be processed
    $volumes = Get-WmiObject -Class Win32_Volume | Where-Object { $_.DriveType -eq 3 -and $_.Name -like "*$MountDirectory*" }
    Write-Host "The following mounted image locations will be processed:"
    foreach ($volume in $volumes) {
        Write-Host "$($volume.Name)"
    }

    # Display the times that will be used for processing
    if ($StartDateTime) {
        if ($StartDateTime) {
            Write-Host "The following start date and time will be used for processing:"
            Write-Host "Start Year: $($parsedStartDateTime.Year)"
            Write-Host "Start Month: $($parsedStartDateTime.ToString("MMM"))"
            Write-Host "Start Day: $($parsedStartDateTime.Day)"
            Write-Host "Start Time: $($parsedStartDateTime.ToString("HH:mm:ss"))"
            Write-Host "Start Time Zone: $($parsedStartDateTime.Offset)"
        }
    }
    else {
        Write-Host "Processsing will start at the beginning of the logs with no start date filter"
    }
    if ($EndDateTime) {
        Write-Host "The following end date and time will be used for processing:"
        Write-Host "End Year: $($parsedEndDateTime.Year)"
        Write-Host "End Month: $($parsedEndDateTime.ToString("MMM"))"
        Write-Host "End Day: $($parsedEndDateTime.Day)"
        Write-Host "End Time: $($parsedEndDateTime.ToString("HH:mm:ss"))"
        Write-Host "End Time Zone: $($parsedEndDateTime.Offset)"
    }
    else {
        Write-Host "Processsing will end at the end of the logs with no end date filter"
    }

    # Confirm that the user wants to continue if requested
    if ($Confirm) {
        while ($true) {
            $response = Read-Host "Do you want to continue? (Y/N)"
            if ($response -imatch "^(y|yes)$") {
                break
            }
            elseif ($response -imatch "^(N|No)$") {
                Write-Host "Exiting..."
                exit
            }
            else {
                Write-Host "Invalid response. Please enter Y or N."
            }
        }
    }

    # Process each directory in the mount directory
    
    # Define the RunHayabusa function to call Hayabasu with the specified arguments
    function RunHayabusa {
        param (
            $HostDir
        )
        
        # Set the path to the event logs directory
        $EvtxDir = Join-Path -Path $MountDirectory -ChildPath "$HostDir\C\Windows\System32\winevt\logs"
        
        # Get the path to the Hayabusa executable
        $HayabusaExe = Get-ChildItem -Path $HayabusaDirectory -Filter "hayabusa*.exe" | Select-Object -ExpandProperty FullName
        
        # Set the arguments for the Hayabusa command
        $Arguments =  "csv-timeline -d `"$($EvtxDir)`" -w --RFC-3339 -U  -o `"$($OutDirectory)\$($HostDir)-all.csv`""
        
        # Add start date and time argument if specified
        If ($StartDateTime) {
            $Arguments += " --timeline-start `"$($parsedStartDateTime.ToString("yyyy-MM-dd HH:mm:ss zzz"))`""
        }
        
        # Add end date and time argument if specified
        If ($EndDateTime) {
            $Arguments += " --timeline-end `"$($parsedEndDateTime.ToString("yyyy-MM-dd HH:mm:ss zzz"))`""
        }
        
        # Start the Hayabusa process and add it to the processes array
        Write-Host "Opening a new window to process $HostDir with $HayabusaExe $Arguments"
        $process = Start-Process -RedirectStandardError "$($OutDirectory)\$($HostDir)-errors.txt"-PassThru -FilePath $HayabusaExe -ArgumentList $Arguments
        Write-Host "Opening a new window to process $HostDir with $HayabusaExe $Arguments"
        $process = Start-Process -RedirectStandardError "$($OutDirectory)\$($HostDir)-errors.txt"-PassThru -FilePath $HayabusaExe -ArgumentList $Arguments
        $script:processes += $process
        
        # Start a separate Hayabusa process for only high and crit detections if specified
        if ($HighCrit) {
            $Arguments = "csv-timeline -d `"$($EvtxDir)`" -w --RFC-3339 -U -m high -o `"$($OutDirectory)\$($HostDir)-high+crit.csv`""
             
            # Add start date and time argument if specified
            If ($StartDateTime) {
                $Arguments += " --timeline-start `"$($parsedStartDateTime.ToString("yyyy-MM-dd HH:mm:ss zzz"))`""
            }
            
            # Add end date and time argument if specified
            If ($EndDateTime) {
                $Arguments += " --timeline-end `"$($parsedEndDateTime.ToString("yyyy-MM-dd HH:mm:ss zzz"))`""
            }
            Write-Host "Starting Hayabusa $HayabusaExe $Arguments"
            $process = Start-Process -RedirectStandardError "$($OutDirectory)\$($HostDir)-errors2.txt" -PassThru -FilePath $HayabusaExe -ArgumentList $Arguments
            $process = Start-Process -RedirectStandardError "$($OutDirectory)\$($HostDir)-errors2.txt" -PassThru -FilePath $HayabusaExe -ArgumentList $Arguments
            $script:processes += $process
        }
    }
    
    Write-Host "Now processing all images mounted in $MountDirectory"
    # Process each mounted volume in the mount directory

    # Get the list of directories in the specified mount directory
    $HostDir = Get-ChildItem -Path "$MountDirectory" -Directory -Name
    # Initialize the processes array
    $script:processes = @()

    # Verify that each folder contains logs folder and call the RunHayabusa function for the ones that do
    $HostDir | ForEach-Object {
    if (Test-Path "$MountDirectory\$_\C\Windows\System32\winevt\logs") {
            RunHayabusa -HostDir $_
    if (Test-Path "$MountDirectory\$_\C\Windows\System32\winevt\logs") {
            RunHayabusa -HostDir $_
        }
        else {
            Write-Host -ForegroundColor Yellow "WARNING: No logs folder found in $MountDirectory$_\C\Windows\System32\winevt\logs. Skipping $_."
        }
    }
        else {
            Write-Host -ForegroundColor Yellow "WARNING: No logs folder found in $MountDirectory$_\C\Windows\System32\winevt\logs. Skipping $_."
        }
    }

    Write-Host "****************************************************************************************************"
    Write-Host "Each new window is processing a task. `nAs each task completes, the associated window will close."
    Write-Host "Until then, the data is still being processed. Please wait..."

    # Monitor processes and update progress bar
    while ($script:processes | Where-Object { $_.HasExited -eq $false }) {
        Start-Sleep -Seconds 1 # Wait before checking again

        # Calculate completed processes
        $completed = ($script:processes | Where-Object { $_.HasExited }).Count
        $total = $script:processes.Count
        $percentComplete = ($completed / $total) * 100

        Write-Progress -Activity "Processing Logs" -Status "Waiting for processing to complete" -PercentComplete $percentComplete
    }
 
    # After all processes have completed, continue...
    # Combine the high and critical detections from each host into a single file for quick reference if requested
    if ($HighCrit) {
        if (Test-Path "$OutDirectory\combined-high+crit.csv") {
        Write-Host -ForegroundColor Red "ERROR: A file named combined-high+crit.csv already exists in the output directory.`nSkipping creation of new combined file."
        }
        else{
            try {
                Get-ChildItem -Path "$OutDirectory" -Filter "*high+crit.csv" | ForEach-Object {

                    Get-Content $_.FullName | Add-Content "$OutDirectory\combined-high+crit.csv"
                }
            }
            catch {
                Write-Host -ForegroundColor Yellow "No high and critical detections found on processed logs."
            }
         }
    }
    # Remove empty error files
    Get-ChildItem -Path $OutDirectory -Include "*-errors.txt", "*-errors2.txt" -Recurse | Where-Object { $_.Length -eq 0 } | Remove-Item -Force
    
    # Report completion
    Write-Host "All instances of Hayabusa have completed processing.`nThe results are stored in $($OutDirectory)"
    Write-Host "If Hayabusa reported problems, errors for each system are reported in files ending in -errors.txt and -errors2.txt."
    Write-Host "You might also found a .\logs folder that contains dated errorlog files.`nYou may now begin your analysis."
    