<#
    .SYNOPSIS
    Uses PowerShell remoting to connect to a defined list of target computers and run Kape with the specified arguments. The output is stored on a network share.

    .DESCRIPTION
    Uses PowerShell remoting to connect to a defined list of target computers and run Kape with the specified arguments. The output is stored on a network share. The script will prompt for a credential with administrator access to the target computers. The script will also prompt for a credential with administrator access to the share server. The server can be configured in advance using Prepare-AutoKapeServer.ps1. A Settings.psd1 configuration file is used to ensure that the same settings are used for both scripts. 
    
    .PARAMETER SettingsFile
    The path to the settings.psd1 file. If not specified, the script will look for the file in the same directory as the script.
    
    .EXAMPLE
    PS> Run-AutoKape.ps1 
    This looks for Settings.psd1 in the same directory as the script and runs Kape on the target computers with the arguments specified in Settings.psd1. 
    
    .EXAMPLE
    PS> Run-AutoKape.ps1 -SettingsFile C:\Users\user\Desktop\Settings.psd1
    This runs Kape on the target computers with the arguments specified in C:\Users\user\Desktop\Settings.psd1.

    .NOTES
    Written by Steve Anson. This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. The GNU General Public License can be found at <https://www.gnu.org/licenses/>.
    
    .LINK
    You can download Kape from here: https://www.kroll.com/en/services/cyber-risk/incident-response-litigation-support/kroll-artifact-parser-extractor-kape
#>

param (
    [string]$SettingsFile = (Join-Path -Path $PWD -ChildPath 'Settings.psd1'),
    [string[]]$TargetComputers,
    [string]$ServerUserName,
    [SecureString]$ServerUserPassword,
    [string]$ServerIPAddress,
    [string]$ServerSubnetMask,
    [string]$DNSServer1,
    [string]$DNSServer2,
    [string]$DefaultGateway,
    [string]$KapeExePath,
    [string]$KapeExeSharedFolderName,
    [string]$DestinationPath,
    [string]$DestinationSharedFolderName,
    [string]$ServerInterfaceName,
    [string]$ServerTranscriptPath,
    [string]$ServerTranscriptName,
    [string]$RunTranscriptPath,
    [string]$RunTranscriptName,
    [string]$KapeArguments    
)

# Import the .psd1 file
if (-not (Test-Path -Path $SettingsFile)) {
    Write-Host -ForegroundColor Red "ERROR: $SettingsFile file not found."
    Write-Host -ForegroundColor Red "Make sure that the file exists or specify a different location with the -SettingsFile parameter."
    Exit 1
}

$variables = Import-PowerShellDataFile -Path $SettingsFile

# Assign values from the imported file to the parameters
$TargetComputers = $variables.TargetComputers
$ServerUserName = $variables.ServerUserName
if ($variables.ServerUserPassword) {
    $ServerUserPassword = ConvertTo-SecureString -String $variables.ServerUserPassword -AsPlainText -Force
}
$ServerIPAddress = $variables.ServerIPAddress
$ServerSubnetMask = $variables.ServerSubnetMask
$DNSServer1 = $variables.DNSServer1
$DNSServer2 = $variables.DNSServer2
$DefaultGateway = $variables.DefaultGateway
$KapeExePath = $variables.KapeExePath
$KapeExeSharedFolderName = $variables.KapeExeSharedFolderName
$KapeExeShare = "\\$ServerIPAddress\$KapeExeSharedFolderName"
$DestinationPath = $variables.DestinationPath
$DestinationSharedFolderName = $variables.DestinationSharedFolderName
$DestinationShare = "\\$ServerIPAddress\$DestinationSharedFolderName"
$ServerInterfaceName = $variables.InterfaceName
$ServerTranscriptPath = $variables.ServerTranscriptPath
$ServerTranscriptName = $variables.ServerTranscriptName
$RunTranscriptPath = $variables.RunTranscriptPath
$RunTranscriptName = $variables.RunTranscriptName
$KapeArguments = $variables.KapeArguments

# Get ServerUserPassword if not provided by the settings file
if (-NOT $ServerUserPassword) {
    Write-Host "Please enter the password for the $ServerUserName account."
    $ServerUserPassword = Read-Host -AsSecureString
}

#Start Transcript
if (-NOT (Test-Path -Path $RunTranscriptPath)) {
    Write-Host "Creating $RunTranscriptPath..."
    New-Item -ItemType Directory -Path $RunTranscriptPath -Force
}

$timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
$DatedTranscriptName = "$RunTranscriptName-$timestamp"
$RunTranscript = Join-Path -Path $RunTranscriptPath -ChildPath $DatedTranscriptName

Start-Transcript -Path $RunTranscript
Write-Host "Script transcript starting at $(Get-Date -Format 'yyyy-MMM-dd HH:mm:ss')."

# Check if the required parameters are empty
if (-not $TargetComputers) {
    Write-Host -ForegroundColor Red "ERROR: TargetComputers is empty in $SettingsFile."
    Write-Host -ForegroundColor Red "Ensure that the file contains a valid list of target computers."
    Exit 1
} else {
    Write-Host "TargetComputers: $TargetComputers"
}

if (-not $ServerUserName) {
    Write-Host -ForegroundColor Red "ERROR: ServerUserName is empty in $SettingsFile."
    Write-Host -ForegroundColor Red "Ensure that the file contains a valid username for the server."
    Exit 1
} else {
    Write-Host "ServerUserName: $ServerUserName"
}

if (-not $ServerUserPassword) {
    Write-Host -ForegroundColor Red "ERROR: ServerUserPassword is empty in $SettingsFile."
    Write-Host -ForegroundColor Red "Ensure that the file contains a valid password for the server."
    Exit 1
} else {
    Write-Host "ServerUserPassword: is set...check the settings file for the password"
}

if (-not $KapeExeSharedFolderName) {
    Write-Host -ForegroundColor Red "ERROR: KapeExeSharedFolderName is empty in $SettingsFile."
    Write-Host -ForegroundColor Red "Ensure that the file contains a valid name for the share where the Kape.exe file is stored."
    Exit 1
} else {
    Write-Host "KapeExeSharedFolderName: $KapeExeSharedFolderName"
}

if (-not $DestinationSharedFolderName) {
    Write-Host -ForegroundColor Red "ERROR: DestinationSharedFolderName is empty in $SettingsFile."
    Write-Host -ForegroundColor Red "Ensure that the file contains a valid name for the share where the output will be stored."
    Exit 1
} else {
    Write-Host "DestinationSharedFolderName: $DestinationSharedFolderName"
}

if (-not $RunTranscriptPath) {
    Write-Host -ForegroundColor Red "ERROR: RunTranscriptPath is empty in $SettingsFile."
    Write-Host -ForegroundColor Red "Ensure that the file contains a valid path to the location where the transcript file will be created."
    Exit 1
} else {
    Write-Host "RunTranscriptPath: $RunTranscriptPath"
}

if (-not $RunTranscriptName) {
    Write-Host -ForegroundColor Red "ERROR: RunTranscriptName is empty in $SettingsFile."
    Write-Host -ForegroundColor Red "Ensure that the file contains a valid base name for the transcript file."
    Exit 1
} else {
    Write-Host "RunTranscriptName: $RunTranscriptName"
}

if (-not $KapeArguments) {
    Write-Host -ForegroundColor Red "ERROR: KapeArguments is empty in $SettingsFile."
    Write-Host -ForegroundColor Red "Ensure that the file contains valid arguments for Kape."
    Exit 1
} else {
    Write-Host "KapeArguments: $KapeArguments"
}

# Create a credential to use on the Kape server
Try {
    $KapeServerCredential = New-Object System.Management.Automation.PSCredential($ServerUserName, $ServerUserPassword)
} catch {
    Write-Host -ForegroundColor Red "ERROR: Failed to create a credential for $ServerUserName. `n$_"
    exit 1
}

# Verify server destination share presence and permissions
try {
    $PsDriveDest = "Dest"
    New-PSDrive -Name $PsDriveDest -PSProvider FileSystem -Root $DestinationShare -Credential $KapeServerCredential -ErrorAction Stop | Out-Null
} catch {
    Write-Host -ForegroundColor Red "ERROR: Failed to authenticate to $DestinationShare. `n$_"
    exit 1
} 


$testFile = "$PSDriveDest`:" + "connect_test_" + (Get-Date -Format "yyyyMMdd_HHmmss") + ".txt"
try {
    New-Item -Path $testFile -ItemType File -ErrorAction Stop | Out-Null
    Remove-Item -Path $testFile -ErrorAction Stop | Out-Null
    Write-Host "Write permission to $DestinationShare is confirmed."
} catch {
    Write-Host -ForegroundColor Red "ERROR: Write to $DestinationShare failed. Check ServerUserName in settings.psd1 and ensure that the designated account has permissions to write to the shared folder.`n$_"
    exit 1
}

# Verify Kape exe share presence and permissions
try {
    $PsDriveKape = "Kape"
    New-PSDrive -Name $PsDriveKape -PSProvider FileSystem -Root $KapeExeShare -Credential $KapeServerCredential -ErrorAction Stop | Out-Null
    Get-ChildItem -Path $PsDriveKape`: -ErrorAction Stop | Out-Null
    Write-Host "Read permission to $KapeExeShare is confirmed."
} catch {
    Write-Host -ForegroundColor Red "ERROR: Failed to authenticate to $KapeExeShare. `n$_"
    exit 1
}


# Check if kape.exe exists in the root of $PsDriveKape and store its hash
try { 
    if (Test-Path -Path "$($PsDriveKape):\kape.exe") {
        Write-Host "kape.exe exists in the root of $PsDriveKape."   
        # Compute the SHA512 hash of kape.exe
        $KapeDrive = Get-PSDrive -Name $PsDriveKape
        $SharedKapeFile = $KapeDrive.Root + "\kape.exe"
        $KapeExeHash = Get-FileHash -Path $SharedKapeFile -Algorithm SHA512 -ErrorAction Stop
        # Store just the hash in $KapeExeHash
        $KapeExeHash = $KapeExeHash.Hash
        Write-Host "The starting SHA512 hash of kape.exe in $KapeExeShare is $KapeExeHash."
    } else {
        Write-Host -ForegroundColor Red "kape.exe does not exist in the root of $PsDriveKape."
        exit 1
    }

} catch {
    Write-Host -ForegroundColor Red "ERROR: Failed to compute the starting hash of kape.exe. `n$_"
    exit 1
}

# Display reminders to the user
Write-Host -ForegroundColor Green "**********IMPORTANT INFORMATION BELOW**********"
Write-Host "The computer where Run-AutoKape.ps1 is run must be able to access the target computers using PowerShell Remoting."
Write-Host "It will also need to have an administrative credential entered during the script execution."
Write-Host "As with any interactive logon, that credential could be exposed if this computer is compromised."
Write-Host "Ideally, a dedicated administrative workstation should be used for this purpose."
Write-Host "Only proceed once you are sure that the above prerequisites have been met."
Write-Host "This script will run remote commands on the target computers with administrator rights."
Write-Host "Please review the script before proceeding to ensure that the actions it will take are appropriate for your environment."

# Confirm that user wants to continue
while ($true) {
    $response = Read-Host "Do you want to continue? (Y/N)"
    if ($response -imatch "^(y|yes)$") {
        break
    }
    elseif ($response -imatch "^(N|No)$") {
        Write-Host "Exiting..."
        exit
    }
    else {
        Write-Host "Invalid response. Please enter Y or N."
    }
}

# Collect the admin credentials for the target computers 
$TargetCredential = Get-Credential -Message "Enter a credential with administrator access for all computers to be processed."

$FailedTargets = @()

# Establish PSSessions to each Target Computer and store them in an array
$PSSessions = @()
$FailedSessions = @()
foreach ($computer in $TargetComputers) {
    try {
        # Create a new PSSession and add it to the array
        $PSSessions += New-PSSession -ComputerName $computer -Credential $TargetCredential -ErrorAction Stop
        Write-Host "PSSession established to $computer."
    } catch {
        Write-Host -ForegroundColor Red "ERROR: Failed to establish a PSSession to $computer. `n$_"
        Write-Host -ForegroundColor Red "ERROR: No further action will be taken on $computer."
        $FailedTargets += $computer
        $FailedSessions += $session
    }
}

# Validate that the credential is valid on all target computers
foreach ($session in $PSSessions) {
    $isAdmin = Invoke-Command -Session $session -ScriptBlock {
        $currentIdentity = [System.Security.Principal.WindowsIdentity]::GetCurrent()
        $principal = New-Object System.Security.Principal.WindowsPrincipal($currentIdentity)
        $principal.IsInRole([System.Security.Principal.WindowsBuiltInRole]::Administrator)
    }

    if (-not $isAdmin) {
        Write-Host -ForegroundColor Red "ERROR: The credential does not have administrator rights on $($session.ComputerName)."
        Write-Host -ForegroundColor Red "ERROR: No further action will be taken on $($session.ComputerName)."
        $FailedTargets += $session.ComputerName
        $FailedSessions += $session
        Remove-PSSession -Session $session
    }
}

# Remove the failed sessions from the $PSSessions array
$PSSessions = $PSSessions | Where-Object { $FailedSessions -notcontains $_ }


# Run Kape
Write-Host "Running Kape on $($PSSessions.Count) target computers with the following syntax:"
Write-Host "$KapeExeShare\kape.exe $KapeArguments"

foreach ($session in $PSSessions) {
    try {
        Invoke-Command -Session $session -ScriptBlock {
            try{
                # Map PsDrive to the destination share on the Kape server
                $PsDriveDest = "Dest"
                New-PSDrive -Name $PsDriveDest -PSProvider FileSystem -Root $using:DestinationShare -Credential $using:KapeServerCredential -ErrorAction Stop | Out-Null
            } catch {
                Write-Host -ForegroundColor Red "ERROR: Failed to connect to $using:DestinationShare from $($session.ComputerName). `n$_"
                Write-Host -ForegroundColor Red "Failed to run Kape on $($session.ComputerName)."
                Write-Host -ForegroundColor Red "Ensure that you have IP connectivity to the server from $($session.ComputerName)."
                Write-Host -ForegroundColor Red "Ensure that no network controls are blocking port 445 (SMB) from $($session.ComputerName) to the server."
                Write-Host -ForegroundColor Red "No further action will be taken on $($session.ComputerName)."
                $FailedTargets += $session.ComputerName
                exit 1
            }
            try {
                # Map PsDrive to the Kape.exe share on the Kape server
                $PsDriveKape = "Exe"
                New-PSDrive -Name $PsDriveKape -PSProvider FileSystem -Root $using:KapeExeShare -Credential $using:KapeServerCredential -ErrorAction Stop | Out-Null
            } catch {
                Write-Host -ForegroundColor Red "ERROR: Failed to connect to $using:KapeExeShare from $($session.ComputerName). `n$_"
                Write-Host -ForegroundColor Red "Failed to run Kape on $($session.ComputerName)."
                Write-Host -ForegroundColor Red "No further action will be taken on $($session.ComputerName)."
                $FailedTargets += $session.ComputerName
                exit 1
            }

            try{
                # Compute the current SHA512 hash of kape.exe on the Kape server
                $CurrentKapeHash = Get-FileHash -Path "$($PsDriveKape):kape.exe" -Algorithm SHA512 -ErrorAction Stop
                $CurrentKapeHash = $CurrentKapeHash.Hash
                # Verify the integrity of kape.exe
                Write-Host "The current SHA512 hash of kape.exe is $CurrentKapeHash."
                if ($CurrentKapeHash -ne $using:KapeExeHash) {
                    Write-Host -Message "ERROR: The hash of kape.exe does not match the expected hash. The integrity of the file cannot be verified."
                    Write-Host -ForegroundColor Red "Failed to run Kape on $($session.ComputerName)."
                    Write-Host -ForegroundColor Red "No further action will be taken on $($session.ComputerName)."
                    $FailedTargets += $session.ComputerName
                    exit 1
                } else {
                    Write-Host "The hash of kape.exe matches the expected hash. The integrity of the file is verified."
                }
            } catch {
                Write-Host -ForegroundColor Red "ERROR: Failed to verify the integrity of kape.exe. `n$_"
                Write-Host -ForegroundColor Red "ERROR: Failed to run Kape on $($session.ComputerName)."
                Write-Host -ForegroundColor Red "ERROR: No further action will be taken on $($session.ComputerName)."
                $FailedTargets += $session.ComputerName
                exit 1
            }
            try{
                # Run Kape
                $DestinationShare = $using:DestinationShare
                & Invoke-Expression "$($PsDriveKape):kape.exe $using:KapeArguments"
            } catch {
                Write-Host -ForegroundColor Red "ERROR: Failed to run Kape on $($session.ComputerName). `n$_"
                Write-Host -ForegroundColor Red "No further action will be taken on $($session.ComputerName)."
                $FailedTargets += $session.ComputerName
            }
        }
    } catch {
        Write-Host -ForegroundColor Red "ERROR: Invoke-Command failed on $($session.ComputerName). `n$_"
        Write-Host -ForegroundColor Red "Failed to run Kape on $($session.ComputerName)."
        Write-Host -ForegroundColor Red "No further action will be taken on $($session.ComputerName)."
        $FailedTargets += $session.ComputerName
    }
}

# Close the PSSessions
foreach ($session in $PSSessions) {
    Remove-PSSession -Session $session
}

# Check if there are any failed targets
if ($FailedTargets) {
    Write-Host -ForegroundColor Red "ERROR: Failed to run Kape on the following target computers:"
    foreach ($failedTarget in $FailedTargets) {
        Write-Host -ForegroundColor Red "- $failedTarget"
    }
    Write-Host -ForegroundColor Red "No further action will be taken on the failed targets."
    Write-Host -ForegroundColor Red "Check the transcript file for more details."
}

# Remove the PsDrives
try {
    Remove-PSDrive -Name $PsDriveKape -ErrorAction Stop
    Remove-PSDrive -Name $PsDriveDest -ErrorAction Stop
}
catch {
    Write-Error "An error occurred while removing the drives: $_"
}

# Stop the transcript
Write-Host "Script transcript stopping at $(Get-Date -Format 'yyyy-MMM-dd HH:mm:ss')."
Stop-Transcript
Write-Host "Execution completed. Closing transcript file."