<#
    .SYNOPSIS
    Undoes many of the changes made by Prepare-AutoKapeServer.ps1. 

    .DESCRIPTION
    Undoes many of the changes made by Prepare-AutoKapeServer.ps1. It deletes the temporary user account, removes the SMB shares and deletes the associated NTFS permissions. It does not modify the network settings or the contents of the Kape executable folder or the Kape results folder. A Settings.psd1 configuration file is used to ensure that the same settings are used for the server configuration and collection scripts. 

    .PARAMETER SettingsFile
    The path to the settings.psd1 file. If not specified, the script will look for the file in the same directory as the script.
    
    .EXAMPLE
    PS> Prepare-AutoKapeServer.ps1
    This looks for Settings.psd1 in the same directory as the script and restores many of the settings on the server to their original values.
    
    .EXAMPLE
    PS> Prepare-AutoKapeServer.ps1 -SettingsFile C:\Users\user\Desktop\Settings.psd1
    This restores many of the settings on the server to their original values using the settings specified in C:\Users\user\Desktop\Settings.psd1. The same Settings.psd1 file must be used with the same values as was used to configure the server with Prepare-AutoKapeServer.ps1.

    .NOTES
    Written by Steve Anson. This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. The GNU General Public License can be found at <https://www.gnu.org/licenses/>.
    
    .LINK
    You can download Kape from here: https://www.kroll.com/en/services/cyber-risk/incident-response-litigation-support/kroll-artifact-parser-extractor-kape
#>

param (
    [string]$SettingsFile = (Join-Path -Path $PWD -ChildPath 'Settings.psd1'),
    [string[]]$TargetComputers,
    [string]$ServerUserName,
    [SecureString]$ServerUserPassword,
    [string]$ServerIPAddress,
    [string]$ServerSubnetMask,
    [string]$DNSServer1,
    [string]$DNSServer2,
    [string]$DefaultGateway,
    [string]$KapeExePath,
    [string]$KapeExeSharedFolderName,
    [string]$DestinationPath,
    [string]$DestinationSharedFolderName,
    [string]$ServerInterfaceName,
    [string]$ServerTranscriptPath,
    [string]$ServerTranscriptName,
    [string]$RunTranscriptPath,
    [string]$RunTranscriptName,
    [string]$KapeArguments    
)

# Check if the current PowerShell session is running as administrator
try {
    $currentPrincipal = [Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()
    if (-NOT $currentPrincipal.IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
        throw 
    }
}
catch {
    Write-Host -ForegroundColor Red "ERROR: This script must be run as Administrator."
    exit 1
}

# Import the .psd1 file
if (-not (Test-Path -Path $SettingsFile)) {
    Write-Host -ForegroundColor Red "ERROR: $SettingsFile file not found."
    Write-Host -ForegroundColor Red "Make sure that the file exists or specify a different location with the -SettingsFile parameter."
    Exit 1
}

$variables = Import-PowerShellDataFile -Path 'settings.psd1'

# Assign values from the imported file to the parameters
$TargetComputers = $variables.TargetComputers
$ServerUserName = $variables.ServerUserName
$ServerUserPassword = ConvertTo-SecureString -String $variables.ServerUserPassword -AsPlainText -Force
$ServerIPAddress = $variables.ServerIPAddress
$ServerSubnetMask = $variables.ServerSubnetMask
$DNSServer1 = $variables.DNSServer1
$DNSServer2 = $variables.DNSServer2
$DefaultGateway = $variables.DefaultGateway
$KapeExePath = $variables.KapeExePath
$KapeExeSharedFolderName = $variables.KapeExeSharedFolderName
$KapeExeShare = "\\$ServerIPAddress\$KapeExeSharedFolderName"
$DestinationPath = $variables.DestinationPath
$DestinationSharedFolderName = $variables.DestinationSharedFolderName
$DestinationShare = "\\$ServerIPAddress\$DestinationSharedFolderName"
$ServerInterfaceName = $variables.InterfaceName
$ServerTranscriptPath = $variables.ServerTranscriptPath
$ServerTranscriptName = $variables.ServerTranscriptName
$RunTranscriptPath = $variables.RunTranscriptPath
$RunTranscriptName = $variables.RunTranscriptName
$KapeArguments = $variables.KapeArguments

#Start Transcript
if (-NOT (Test-Path -Path $ServerTranscriptPath)) {
    Write-Host "Creating $ServerTranscriptPath..."
    New-Item -ItemType Directory -Path $ServerTranscriptPath -Force
}

$timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
$DatedTranscriptName = "$ServerTranscriptName-$timestamp"
$ServerTranscript = Join-Path -Path $ServerTranscriptPath -ChildPath $DatedTranscriptName

Start-Transcript -Path $ServerTranscript
Write-Host "Script transcript starting at $(Get-Date -Format 'yyyy-MMM-dd HH:mm:ss')."


# Explanation of upcoming actions
Write-Host "The following actions will be performed by this script:"
Write-Host "1. The local user named '$ServerUserName' will be deleted."
Write-Host "2. The SMB share named '$DestinationSharedFolderName' that shares '$DestinationPath' will be deleted."
Write-Host "3. The SMB share named '$KapeExeSharedFolderName' that shares '$KapeExePath' will be deleted."
Write-Host "4. The NTFS permissions for the user named '$ServerUserName' on the '$KapeExePath' and '$DestinationPath' will be removed."
Write-Host "Please note that these actions will modify the system configuration and may affect its behavior."
Write-Host "This script is not able to undo the changes once they have been made."


# Confirm that user wants to continue
Write-Host "Please confirm that you wish the script to continue and make the changes above."
while ($true) {
    $response = Read-Host "Do you want to continue? (Y/N)"
    if ($response -imatch "^(y|yes)$") {
        break
    }
    elseif ($response -imatch "^(N|No)$") {
        Write-Host "Exiting..."
        exit
    }
    else {
        Write-Host "Invalid response. Please enter Y or N."
    }
}

# Delete the shares
try {
    Write-Host "Deleting the share $DestinationSharedFolderName..."
    Remove-SmbShare -Name $DestinationSharedFolderName -Force
} catch {
    Write-Host "Error deleting share $DestinationSharedFolderName `n$_"
}

try {
    Write-Host "Deleting the share $KapeExeSharedFolderName..."
    Remove-SmbShare -Name $KapeExeSharedFolderName -Force
} catch {
    Write-Host "Error deleting share $KapeExeSharedFolderName `n$_"
}

# Remove NTFS permissions
try {
    Write-Host "Removing NTFS permissions for $ServerUserName on $KapeExePath..."
    $acl = Get-Acl -Path $KapeExePath
    $acl.Access | Where-Object { $_.IdentityReference -eq "$env:COMPUTERNAME\$ServerUserName" } | ForEach-Object { $acl.RemoveAccessRule($_) }
    Set-Acl -Path $KapeExePath -AclObject $acl
} catch {
    Write-Host "Error removing NTFS permissions on $KapeExePath `n$_"
}

try {
    Write-Host "Removing NTFS permissions for $ServerUserName on $DestinationPath..."
    $acl = Get-Acl -Path $DestinationPath
    $acl.Access | Where-Object { $_.IdentityReference -eq "$env:COMPUTERNAME\$ServerUserName" } | ForEach-Object { $acl.RemoveAccessRule($_) }
    Set-Acl -Path $DestinationPath -AclObject $acl
} catch {
    Write-Host "Error removing NTFS permissions on $DestinationPath `n$_"
}

# Delete the user
try {
    Write-Host "Deleting the user $ServerUserName..."
    Remove-LocalUser -Name $ServerUserName
} catch {
    Write-Host "Error deleting user: $_"
}

# Remove firewall rules
try {
    Write-Host "Removing firewall rules beginning with 'KAPE: Allow inbound TCP port 445 from'..."
    Get-NetFirewallRule | Where-Object { $_.DisplayName -like "KAPE: Allow inbound TCP port 445 from*" } | Remove-NetFirewallRule
} catch {
    Write-Host "Error removing firewall rules: $_"
}

Write-Host "There are several things that this script does not modify. If these need changed, please do so manually."
Write-Host "1. The IP address"
Write-Host "2. The subnet mask"
Write-Host "3. The DNS servers"
Write-Host "4. The default gateway"
Write-Host "5. The contents of $KapeExePath" 
Write-Host "6. The contents of $DestinationPath"
Write-Host "7. The power settings (currently set to never sleep or hibernate)"


Write-Host "Would you like to review the current network settings? This will run 'ipconfig /all'."
$response = Read-Host "Please enter Y or Yes to proceed or any other key to skip."

if ($response -eq 'Y' -or $response -eq 'y' -or $response -eq 'Yes' -or $response -eq 'yes') {
    ipconfig /all
}

# Stop the transcript
Write-Host "Script stopping at $(Get-Date -Format 'yyyy-MMM-dd HH:mm:ss')."
Stop-Transcript
Write-Host "Transcript completed. Closing transcript file."